/** \file editor.c
 * \brief <      >
 * \par
 * <       >
 * \par \author ARV \par
 * \note <  >
 * \n :
 * \n \date	25.02.2015
 * \par
 * \version <>.	\par
 * Copyright 2015  ARV. All rights reserved.
 * \par
 *   :\n
 * 	-# Atmel Toolchain 3.4.5    
 *
 */

#include <avr/io.h>
#include <avr_helper.h>
#include <avr/pgmspace.h>
#include <string.h>
#include <stdlib.h>

#include "config.h"
#include "editor.h"
#include "display.h"
#include "buttons.h"
#include "timekeep.h"
#include "program.h"
#include "memory.h"

#define WD_LEN	2
#define MN_LEN  3

typedef enum {
	EXIT_LEFT = 1,
	EXIT_RIGHT = 2,
} exit_mode;

//   .   ,       !
static __flash const uint8_t approved[] = {
//      0        1        2         3         4        5         6         7        8
	TOC_HOUR, TOC_MIN, TOC_SEC, TOC_QUART, TOC_DAY, TOC_MON, TOC_YEAR, TOC_WDAY, TOC_BR0,
//   9    10   11   12   13   14   15   16   17   18    19       20       21       22
	'0', '1', '2', '3', '4', '5', '6', '7', '8', '9', TOC_BR1, TOC_AND, TOC_OR, TOC_INTERVAL,
//     23
	TOC_EQUAL
};

#define FIRST_VAR	0
#define LAST_VAR	7
#define NCHR		0xFF

//  
typedef struct {
	uint8_t	ok;			//     
	uint8_t	id_start;	//    
	uint8_t	id_end;		//    
} smart_rule;

#define RULES_CNT	16

// N -  
// V -  
// * -  
// . -  
//    

static __flash const
smart_rule rules[RULES_CNT] ={
	//               ok		id_start	id_end
	/* 0  "..." */  {0, 	0, 			8},
	/* 1  "..N" */  {1, 	NCHR, 		NCHR},
	/* 2  "**V" */  {0, 	23, 		23},
	/* 3  "*W=" */  {1, 	10,			16},
	/* 4  "*q=" */  {1, 	9, 			12},
	/* 5  "*V=" */  {1, 	9, 			18},
	/* 6  "*=N" */  {1, 	9, 			22},
	/* 7  "=NN" */  {1, 	19, 		22},
	/* 8  "**-" */  {0, 	9, 			19},
	/* 9  "*-N" */  {1, 	9, 			21},
	/* 10 "-NN" */  {1, 	19, 		21},
	/* 11 "**," */  {0, 	0,  		8},
	/* 12 "**\" */  {0, 	0,  		8},
	/* 13 "**)" */  {1, 	19, 		21},
	/* 14 "q=N" */  {1, 	19, 		22},
	/* 15 "*D=" */  {1,		10,			12}
};

//     
static void sprn_dec999(uint8_t *s, uint8_t d){
	uint8_t t = d / 100;
	*s++ = t + '0';
	*s++ = (d - t*100) / 10 + '0';
	*s++ = (d - t*100) % 10 + '0';
}

//       ""
static uint8_t is_var(uint8_t c){
	for(uint8_t i=FIRST_VAR; i<=LAST_VAR; i++){
		if(c == approved[i]) return 1;
	}
	return 0;
}

//      
static uint8_t is_dig(uint8_t c){
	return (c >= '0') && (c <= '9');
}

/**    .
 *    3    ,    
 *       .    !
 * @param pos   
 * @param s  
 * @return   
 */
static uint8_t get_rule_id(uint8_t pos, uint8_t *s){
	if(pos == 0)
		return 0;
	if(is_dig(s[pos-1]) && (pos == 1))
		return 1;
	if(is_var(s[pos-1]))
		return 2;
	//      
	if((pos >= 2) && (s[pos-2] == TOC_WDAY) && (s[pos-1] == TOC_EQUAL))
		return 3;
	if((pos >= 3) && (s[pos-3] == TOC_WDAY) && (s[pos-2] == TOC_EQUAL) && is_dig(s[pos-1]))
		return 7;
	if((pos >= 4) && (s[pos-4] == TOC_WDAY) && (s[pos-3] == TOC_EQUAL) && is_dig(s[pos-2])
		&& (s[pos-1] == TOC_INTERVAL))
		return 3;
	if((pos >= 5) && (s[pos-5] == TOC_WDAY) && (s[pos-4] == TOC_EQUAL) && is_dig(s[pos-3])
		&& (s[pos-2] == TOC_INTERVAL) && is_dig(s[pos-1]))
		return 10;
	//     
	if((pos >= 2) && (s[pos-2] == TOC_DAY) && (s[pos-1] == TOC_EQUAL))
		return 15;
	if((pos >= 4) && (s[pos-4] == TOC_DAY) && (s[pos-3] == TOC_EQUAL) && is_dig(s[pos-2])
		&& (s[pos-1] == TOC_INTERVAL))
		return 15;
	if((pos >= 4) && (s[pos-4] == TOC_DAY) && (s[pos-3] == TOC_EQUAL) && is_dig(s[pos-2])
		&& is_dig(s[pos-1]))
		return 7;
	if((pos >= 5) && (s[pos-5] == TOC_DAY) && (s[pos-4] == TOC_EQUAL) && is_dig(s[pos-3])
		&& (s[pos-2] == TOC_INTERVAL) && is_dig(s[pos-1]))
		return 15;
	//  
	if((pos >= 2) && (s[pos-2] == TOC_QUART) && (s[pos-1] == TOC_EQUAL))
		return 4;
	if((pos >= 2) && is_var(s[pos-2]) && (s[pos-1] == TOC_EQUAL))
		return 5;
	if((pos >= 3) && (s[pos-3] == TOC_QUART) && (s[pos-2] == TOC_EQUAL) && is_dig(s[pos-1]))
		return 14;
	if((pos >= 2) && (s[pos-2] == TOC_EQUAL) && is_dig(s[pos-1]))
		return 6;
	if((pos >= 3) && (s[pos-3] == TOC_EQUAL) && is_dig(s[pos-2]) && is_dig(s[pos-1]))
		return 7;
	if((s[pos-1] == TOC_INTERVAL))
		return 8;
	if((pos >= 2) && (s[pos-2] == TOC_INTERVAL) && is_dig(s[pos-1]))
		return 9;
	if((pos >= 3) && (s[pos-3] == TOC_INTERVAL) && is_dig(s[pos-2]) && is_dig(s[pos-1]))
		return 10;
	if((s[pos-1] == TOC_AND))
		return 11;
	if((s[pos-1] == TOC_OR))
		return 12;
	if((s[pos-1] == TOC_BR1))
		return 13;
	return 11; // 11-   
}

//    
static uint8_t find_approved(uint8_t c){
	for(uint8_t i=0; i<sizeof(approved); i++)
		if(c==approved[i]) return i;
	//      ! 
	return 0;
}

//   
static uint8_t check_syntax(uint8_t brackets, uint8_t ri, uint8_t l, uint8_t pos, uint8_t *s){
	return (brackets == 0) &&	//     
			(rules[ri].ok) &&	//     
			(l != 0) &&			//     
			//         
			(s[l-1] != TOC_AND) &&
			(s[l-1] != TOC_INTERVAL) &&
			(s[l-1] != TOC_OR) &&
			(s[l-1] != TOC_EQUAL)
			;
}

/**       .
 *
 * @param chanel  
 * @param pos     
 * @param len   
 * @param check_syntax     
 * @param can_lt 0,     
 * @param can_rt 0,     
 * @param can_up 0,     
 */
static void print_info(uint8_t chanel, uint8_t pos, uint8_t len, uint8_t check_syntax, uint8_t can_lt, uint8_t can_rt, uint8_t can_up){
	uint8_t tmp[17];

	memset(tmp, ' ', 16);
	set_cursor_x(0);

	tmp[0] = 'K';
	itoa(chanel+1, (char*)&tmp[1], 10);
	tmp[2] = ' ';
	sprn_dec999(&tmp[3], pos);
	tmp[6] = '/';
	sprn_dec999(&tmp[7], len);

	tmp[12] = check_syntax ? ' ' : 'X';
	tmp[13] = can_lt ? '<' : ' ';
	tmp[14] = can_up ? 'I' : ' ';
	tmp[15] = can_rt ? '>' : ' ';
	tmp[16] = 0;

	display_prn(tmp);
}

//     
static uint8_t calc_brackets(uint8_t *s){
	uint8_t cnt = 0;
	while(*s){
		if(*s == '(')
			cnt++;
		else if(*s == ')') cnt--;
		s++;
	}
	return cnt;
}

/**    .
 *        .   
 * ,          , ..  
 *    ,    "" .
 *         ,     
 *  (    ),        
 *   .
 * @param chanel   
 * @param dst  
 * @return   #EDIT_OK  #EDIT_CANCEL
 */
edit_result smart_edit(uint8_t chanel, uint8_t *dst){
	uint8_t len, pos = 0;	//  
	uint8_t firstpos = 0;	//     
	uint8_t itempos = NCHR;	//       
	uint8_t rule_id = 0;	//    
	uint8_t key;			//   
	uint8_t br_cnt = 0;		//   
	uint8_t c_ok;			//   

	while(1){
		//  
		visible_cursor(0);
		//    
		len = strlen((char*)dst);
		//      
		if(len){
			if(pos > len) pos = len;
		} else {
			pos = 0;
		}
		//    
		if(pos >= DISPLAY_WIDTH){
			firstpos = pos - DISPLAY_WIDTH +1;
		} else {
			firstpos = 0;
		}
		//   
		rule_id = get_rule_id(pos, dst);
		//     ,  
		if(dst[pos] == 0){
			itempos = rules[rule_id].id_start;
		}
		//   ,  
		if((dst[pos] == 0) && (approved[itempos] == TOC_EQUAL)){
			dst[pos] = approved[itempos];
		}
		//  ,   
		if(pos == len){
			dst[pos+1] = 0;
			len++;
		}
		//   
		br_cnt = calc_brackets(dst);
		//  
		c_ok = check_syntax(br_cnt, rule_id, len, pos, dst);
		//   
		set_cursor_y(0);
		print_info(chanel, pos+1, len, c_ok, dst[pos] || (pos > 0), (len < MAX_SOURCE_LEN) && (dst[pos]), rules[rule_id].id_start != rules[rule_id].id_end);
		//    
		set_cursor_y(1);
		display_nprn(dst+firstpos);
		//     
		set_cursor_x(pos - firstpos);
		//  
		visible_cursor(1);

		//   
		do{
			key = get_key();
			switch(key){
			case K_LT:
				dst[pos] = 0;
				if(pos) pos--;
				//   
				itempos = find_approved(dst[pos]);
				break;
			case K_RT:
				if(len == MAX_SOURCE_LEN) break;
				if(pos < len) pos++;
				if(pos < len){
					//    -   
					itempos = find_approved(dst[pos]);
				} else {
					//       -  
					itempos = NCHR;
				}
				break;
			case K_OK:
				if(c_ok){
					visible_cursor(0);
					return EDIT_OK;
				}
				break;
			case K_UP:
				//      
				do{
					if(itempos < rules[rule_id].id_end)
						itempos++;
					else
						itempos = rules[rule_id].id_start;
					//        
				} while((approved[itempos] == TOC_BR1) && (br_cnt == 0));
				break;
			case K_DN:
				//      
				do{
					if(itempos > rules[rule_id].id_start)
						itempos--;
					else
						itempos = rules[rule_id].id_end;
					//        
				} while((approved[itempos] == TOC_BR1) && (br_cnt == 0));
				break;
			case K_CANCEL:
				visible_cursor(0);
				return EDIT_CANCEL;
			}
		} while(key == K_NONE);
		//      ,   
		if(itempos != NCHR)
			dst[pos] = approved[itempos];
	}
}

///    
static fchar wdlist[] = "";
///   
static fchar monlist[] = "";

#define DEFAULT_CASE() \
	case K_LT:	return EDIT_PREV; \
	case K_RT:	return EDIT_NEXT; \
	case K_OK:	return EDIT_OK

/**   .
 *          .
 * @param len   
 * @param list_len    
 * @param list_id    
 * @param list  
 * @return   
 */
static edit_result edit_list(uint8_t len,		//   
		              uint8_t list_len, // -   
		              uint8_t *list_id,	//     
		              fchar *list		//   
		              ){
	uint8_t key;
	uint8_t old_id = *list_id;

	while(1){
		visible_cursor(0);
		// 
		display_fnprn(list+(*list_id*len),len);
		visible_cursor(1);
#if defined(AUTO_CANCEL_SEC_DELAY)
		auto_cancel = AUTO_CANCEL_SEC_DELAY;
#endif
		do{
			//   
			key = get_key();
#if defined(AUTO_CANCEL_SEC_DELAY)
			if(key != K_NONE)
				auto_cancel = AUTO_CANCEL_SEC_DELAY;
			else
				if(!auto_cancel) key = K_CANCEL;
#endif
			switch(key){
			case K_UP:
				//  -    
				(*list_id)++;				// 
				if(*list_id >= list_len){	//    ,
					*list_id = 0;		//    
				}
				break;
			case K_DN:
				//  -    
				if(*list_id == 0){			//    ,
					*list_id = list_len - 1; //   
				} else {
					(*list_id)--;			//      -  
				}
				break;
			case K_CANCEL:
				//  
				*list_id = old_id;
				//    
				return EDIT_CANCEL;
			DEFAULT_CASE();
			}
		}while(key == K_NONE);
	}
}

/**   .
 *       .
 * @param wd   
 * @return  
 */
static edit_result edit_weekday(uint8_t *wd){
	return edit_list(WD_LEN,7,wd,wdlist);
}

/**  .
 *      .
 * @param mm  
 * @return  
 */
static edit_result edit_month(uint8_t *mm){
	return edit_list(MN_LEN,12,mm,monlist);
}

static void prn_dec99(uint8_t d, uint8_t fill){
	uint8_t buf[3];
	uint8_t l=0;

	if(d > 99) d=99;
	if(d < 10){
		l = 1;
		buf[0] = fill;
	}
	itoa(d, (char*)buf+l, 10);
	display_prn(buf);
}

static edit_result edit_dec99(uint8_t *d, uint8_t min, uint8_t max, char fill){
	uint8_t old = *d;
	uint8_t key;

	while(1){
		visible_cursor(0);
		prn_dec99(*d,'0');
		visible_cursor(1);
#if defined(AUTO_CANCEL_SEC_DELAY)
		auto_cancel = AUTO_CANCEL_SEC_DELAY;
#endif
		do{
			key = get_key();
#if defined(AUTO_CANCEL_SEC_DELAY)
			if(key != K_NONE)
				auto_cancel = AUTO_CANCEL_SEC_DELAY;
			else
				if(!auto_cancel) key = K_CANCEL;
#endif
			switch(key){
			case K_CANCEL:
				*d = old;
				return EDIT_CANCEL;
			case K_UP:
				if(*d < max)	(*d)++;
				else			*d = min;
				break;
			case K_DN:
				if(*d > min)	(*d)--;
				else			*d = max;
				break;
			DEFAULT_CASE();
			}
		} while (key == K_NONE);
	}
}

/*********************************************************
 *    0123456789012345 -    LCD
 *     25  2015,
 *        09:13.00
 *********************************************************/

void show_time(timedate *td){
	set_cursor_x(4);
	prn_dec99(td->hour, '0');
	set_cursor_x(6);
	display_putc(':');
	set_cursor_x(7);
	prn_dec99(td->min, '0');
	set_cursor_x(9);
	display_putc('.');
	set_cursor_x(10);
	prn_dec99(td->sec, '0');
}


void show_date(timedate *td){
	set_cursor_x(1);
	prn_dec99(td->d + 1, '0');
	set_cursor_x(4);
	display_fnprn(monlist + td->m * MN_LEN, MN_LEN);
	set_cursor_x(8);
	prn_dec99(20, '0');
	set_cursor_x(10);
	prn_dec99(td->y, '0');
	set_cursor_x(12);
	display_putc(',');
	set_cursor_x(13);
	display_fnprn(wdlist + td->wd * WD_LEN, WD_LEN);
}

typedef enum {
	STATE_DAY,
	STATE_MON,
	STATE_YEAR,
	STATE_WDAY,
	STATE_HOUR,
	STATE_MIN,
	STATE_SEC,
	//
	STATE_LAST
} edit_state;

edit_result edit_timedate(timedate *td){
	edit_state state = STATE_DAY;
	edit_result result;

	clrscr();
	//  
	show_date(td);
	//  
	set_cursor_y(1);
	show_time(td);

	td->d++; // !!!!  !!!
	while(1){
		switch(state){
		case STATE_DAY: set_cursor_x(1); set_cursor_y(0);
			result = edit_dec99(&td->d, 1, get_month_len(td), '0');
			break;
		case STATE_MON: set_cursor_x(4); set_cursor_y(0);
			result = edit_month(&td->m);
			break;
		case STATE_YEAR: set_cursor_x(10); set_cursor_y(0);
			result = edit_dec99(&td->y, 15, 99, '0');
			break;
		case STATE_WDAY: set_cursor_x(13); set_cursor_y(0);
			result = edit_weekday(&td->wd);
			break;
		case STATE_HOUR: set_cursor_x(4); set_cursor_y(1);
			result = edit_dec99(&td->hour, 0, 23, '0');
			break;
		case STATE_MIN: set_cursor_x(7); set_cursor_y(1);
			result = edit_dec99(&td->min, 0, 59, '0');
			break;
		case STATE_SEC: set_cursor_x(10); set_cursor_y(1);
			result = edit_dec99(&td->sec, 0, 59, '0');
			break;
		default:
			break;
		}

		switch(result){ //  Warning -      result  
		//    - warning  
		case EDIT_OK:
		case EDIT_CANCEL:
			td->d--; // !!!  !!!
			clrscr();
			return result;
		case EDIT_NEXT:
			if(state < (STATE_LAST-1))	state++;
			else						state = STATE_DAY;
			break;
		case EDIT_PREV:
			if(state > STATE_DAY)		state--;
			else						state = STATE_LAST-1;
			break;
		}
	}
}

static const
__flash char no_str[] = "----------------";

edit_result view_chanel(uint8_t *ch){
	uint8_t str[MAX_SOURCE_LEN+1];
	uint8_t tmp[17];
	uint8_t firstpos = 0;
	uint8_t len, key;
	uint16_t fm = get_free_memory();

	while(1){
		start_memory_chanel(*ch);
		decompose(str);
		len = strlen((char*)str);
		set_cursor_y(0);
		set_cursor_x(0);

		memset(tmp, ' ', 16);
		tmp[0] = 'K';
		itoa((*ch)+1, (char*)&tmp[1], 10);
		tmp[2] = ' ';
		tmp[3] = 'L'; tmp[4] = ':';
		sprn_dec999(&tmp[5], len);
		tmp[9] = 'F'; tmp[10] = ':';
		itoa(fm, (char*)&tmp[11], 10);
		display_nprn(tmp);
		set_cursor_y(1);
		set_cursor_x(0);
		if(len)
			display_nprn(str+firstpos);
		else
			display_fnprn(no_str, DISPLAY_WIDTH);

		do{
			key = get_key();
			switch(key){
			case K_OK:
				return EDIT_OK;
			case K_CANCEL:
				return EDIT_CANCEL;
			case K_UP | K_DN:
				//   
				clear_memory();
				break;
			case K_UP:
				if(*ch < CHANEL_NUM-1)
					(*ch)++;
				else
					*ch = 0;
				firstpos = 0;
				break;
			case K_DN:
				if(*ch > 0)
					(*ch)--;
				else
					*ch = CHANEL_NUM-1;
				firstpos = 0;
				break;
			case K_LT:
				if(firstpos) firstpos--;
				break;
			case K_RT:
				if(len-firstpos > DISPLAY_WIDTH)
					firstpos++;
				break;
			}
		} while (key != K_NONE);
	}
}

edit_result edit_chanel(uint8_t ch){
	uint8_t str[MAX_SOURCE_LEN+1];
	uint8_t sz;

	start_memory_chanel(ch);
	decompose(str);
	if(smart_edit(ch, str) == EDIT_OK){
		sz = compose(str);
		save_chanel(ch, sz, str);
		return EDIT_OK;
	}
	return EDIT_CANCEL;
}
