/** \file main.c
 * \brief Timer-repeater up to 60 min
 * \par \author ARV \par
 * \note Specially for Manoj Soorya
 * \n \date	16  2016 .
 * \par
 * \version 0.1	\par
 * Copyright 2015  ARV. All rights reserved.
 */

#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/pgmspace.h>
#include <avr/eeprom.h>
#include <util/delay.h>
#include <util/atomic.h>
#include "avr_helper.h"
#include "config.h"
#include "leds7.h"

/// modes of operation
enum{
	MODE_SET,
	MODE_RUN
};

/// EEPROM cell for timer settings
static EEMEM uint8_t e_timer_value;
/// active timer settings
static volatile uint8_t timer_value;
/// active mode
static volatile uint8_t mode;
/// blink mode status
static volatile uint8_t blink;

/// "screen memory"
static volatile uint8_t SCR[LED_COUNT];
/// symbols for 7-seg leds
static PROGMEM uint8_t digits[] = {d_0, d_1, d_2, d_3, d_4, d_5, d_6, d_7, d_8, d_9};

/** print dec value on 7-seg led display
 *
 * @param data output value
 */
static void print_dec(uint8_t data){
	for(uint8_t i=LED_COUNT; i; i--){
		SCR[i-1] = pgm_read_byte(&digits[data % 10]);
		data /= 10;
	}
}

/** TIMER0 - dynamic indication
 *
 */
ISR(TIMER0_OVF_vect){
	static uint8_t pos;

	// set off all indicators
#if common_anode == 1
	PORT(LED_COM_PORT) &= ~LED_COM_ALL;
#else
	PORT(LED_COM_PORT) |= LED_COM_ALL;
#endif

	// calculate next position
	if(++pos >= LED_COUNT) pos = 0;
	// output symbol
	PORT(LED_SEG_PORT) = SCR[pos];
	// blink mode
	if(!blink){
		uint8_t tmp;
		// calculate next common line
		switch(pos){
		case 0:
			tmp = LED_COM_0;
			break;
		case 1:
			tmp = LED_COM_1;
			break;
		}
		// set on current indicator
#if common_anode == 1
	PORT(LED_COM_PORT) |= tmp;
#else
	PORT(LED_COM_PORT) &= ~tmp;
#endif
	}
}

/** TIMER2 - time conter
 * async mode!
 */
ISR(TIMER2_OVF_vect){
	static uint8_t cnt = 4;
	static uint8_t sec = 60;

	if(mode == MODE_RUN){
		blink = 0;
		if(!--cnt){
			cnt = 4;
			if(!--sec){
				sec = 60;
				if(!--timer_value){
					timer_value = eeprom_read_byte(&e_timer_value);
					PORT(LOAD_PORT) ^= _BV(LOAD_PIN);
				}
			}
			PORT(MODE_PORT) ^= _BV(MODE_PIN);
		}
	} else {
		sec = 60;
		blink ^= 1;
	}
	print_dec(timer_value);
}

/** initialization of all peripheries
 *
 */
INIT(7){
	// PORTS
	DDR(LED_COM_PORT) |= LED_COM_ALL;
	DDR(LED_SEG_PORT) = 255;
	PORT(BUTTON_PORT) |= BUTTON_ANY;
	DDR(LOAD_PORT) |= _BV(LOAD_PIN);
	DDR(MODE_PORT) |= _BV(MODE_PIN);
	// TIMER0
	TIMSK = _BV(TOIE0);
	TCCR0 = TIMER_CLK_DIV_8;
	// TIMER2
	ASSR = _BV(AS2);				//    2
	while(ASSR != _BV(AS2));		// ,  
	TIFR = _BV(TOV2);				//  
	TIMSK |= _BV(TOIE2);			//    
	TCCR2 = _BV(CS21) | _BV(CS20);	//  32
}

/** Start timer
 * by START/SET button pressed
 * running mode on
 */
static void timer_start(void){
	ATOMIC_BLOCK(ATOMIC_RESTORESTATE){
		eeprom_update_byte(&e_timer_value, timer_value);
		_delay_ms(10);
	}
	PORT(LOAD_PORT) |= _BV(LOAD_PIN);
	mode = MODE_RUN;
}

/** Stop timer
 * settings mode on
 */
static void timer_stop(void){
	mode = MODE_SET;
	PORT(LOAD_PORT) &= ~_BV(LOAD_PIN);
	timer_value = eeprom_read_byte(&e_timer_value);
	if((timer_value < MIN_TIMER_VALUE) || (timer_value > MAX_TIMER_VALUE)){
		timer_value = MIN_TIMER_VALUE;
		ATOMIC_BLOCK(ATOMIC_RESTORESTATE){
			eeprom_update_byte(&e_timer_value, timer_value);
			_delay_ms(10);
		}
	}
}

/** get button state
 *
 * @return button code
 */
static uint8_t get_btn(void){
	uint8_t tmp = ~PIN(BUTTON_PORT) & BUTTON_ANY;
	_delay_ms(15);
	if(tmp == (~PIN(BUTTON_PORT) & BUTTON_ANY))
		return tmp;
	else
		return 0;
}

/** main func
 *
 * @return nothing
 */
MAIN(){
	uint8_t btn;
	timer_stop();
	sei();
	while(1){
		// wait for button pressed - for speed
		do {
			btn = get_btn();
		} while (btn == 0);

		// processing of pressed button
		switch(btn){
		case BUTTON_ST:
			if(mode == MODE_RUN)
				timer_stop();
			else
				timer_start();
			// no repeat of START/STOP button!
			while(btn == get_btn());
			break;

		case BUTTON_UP:
			if(mode == MODE_RUN) break;
			if(++timer_value > MAX_TIMER_VALUE) timer_value = MIN_TIMER_VALUE;
			break;

		case BUTTON_DN:
			if(mode == MODE_RUN) break;
			if(--timer_value < MIN_TIMER_VALUE) timer_value = MAX_TIMER_VALUE;
			break;
		}
		// wait releasing or auto repeat button
		for(uint8_t i=0; get_btn() && (i<10); i++){
			_delay_ms(50);
		}
	}
}
